# 定义大学人事管理系统需要的人事信息类
# 包括表示各种教职员工的类，表示各种类别学生的类

# 首先定义一个通用的人员类

import datetime

class PersonTypeError(TypeError):
    pass


class PersonValueError(ValueError):
    pass


class Person:
    def __init__(self, name, sex, birthday):
        if not (isinstance(name, str) and
                sex in ("女", "男")):
            raise PersonValueError(name, sex)
        try:
            birth = datetime.date(*birthday)  # 生成一个日期对象
        except:
            raise PersonValueError("Wrong date:", birthday)

        self._name = name
        self._sex = sex
        self._birthday = birth

    def name(self): return self._name
    def sex(self): return self._sex
    def birthday(self): return self._birthday
    def age(self): return (datetime.date.today().year -
                           self._birthday.year)

    def set_name(self, name): # 修改名字
        if not isinstance(name, str):
            raise PersonValueError("set_name", name)
        self._name = name

    def __lt__(self, another):
        if not isinstance(another, Person):
            raise PersonTypeError("Person Comparison.")
        return self._name < another._name

    def __str__(self):
        return " ".join((self._name, self._sex,
                         str(self._birthday))
                       )

    def details(self):
        return ", ".join(("姓名: " + self._name,
                          "性别: " + self._sex,
                          "出生日期: " + str(self._birthday))
                        )


class UniPerson(Person):
    _pnum = 0
    
    def __init__(self, name, sex, birthday, ident):
        super().__init__(name, sex, birthday)
        self._id = ident
        UniPerson._pnum += 1 # 实例计数

    def id(self): return self._id

    def __lt__(self, another):
        if not isinstance(another, UniPerson):
            raise PersonTypeError("Person Comparison.")
        return self._id < another._id

    @classmethod
    def num(cls): return UniPerson._pnum
    
    def __str__(self):
        return " ".join((self._id, super().__str__()))

    def details(self):
        return ", ".join(("ID: " + self._id,
                          super().details())
                        )


class Student(UniPerson):
    _id_num = 0

    @classmethod
    def _id_gen(cls):  # 实现学号生成规则
        cls._id_num += 1
        year = datetime.date.today().year
        return "1{:04}{:05}".format(year, cls._id_num)
    
    def __init__(self, name, sex, birthday, department):
        super().__init__(name, sex, birthday,
                         Student._id_gen())
        self._department = department
        self._enroll_date = datetime.date.today()
        self._courses = {} # 一个空字典

    def set_course(self, course_name):
        self._courses[course_name] = None

    def set_score(self, course_name, score):
        self._courses[course_name] = score

    def department(self): return self._department
    def en_year(self): return self._enroll_date.year
    def scores(self): return [(cname, self._courses[cname])
                              for cname in self._courses]

    def details(self):
        return ", ".join((super().details(),
                          "院系: " + self.department(),
                          "入学: " + str(self.en_year()),
                          "成绩: " + str(self.scores()))
                        )

    # 其他必要的方法定义从略


class Undergraduate(Student):
    def __init__(self, name, sex, birthday, department, sclass):
        super().__init__(name, sex, birthday, department)
        self._class = sclass

    def details(self):
        return ", ".join((super().details(),
                          "班级: " + self._sclass)
                        )


class MasterStudent(Student):
    def __init__(self, name, sex, birthday, department):
        super().__init__(name, sex, birthday, department)
        self._id = "5" + self._id[1:]
        self._supervisor = "Undetermined" # 导师未定
        self._field = "Undetermined"      # 研究领域未定

    def details(self):
        return ", ".join((super().details(),
                          "导师: " + self._supervisor,
                          "方向: " + self._field)
                        )


class DoctorStudent(MasterStudent):
    def __init__(self, name, sex, birthday, department,
                 field, supervisor):
        super().__init__(name, sex, birthday, department)
        self._id = "8" + self._id[1:]
        self._supervisor = supervisor
        self._field = field


class Staff(UniPerson):
    _id_num = 0
    
    @classmethod
    def _id_gen(cls, birthday):  # 实现职工号生成规则
        Staff._id_num += 1
        birth_year = datetime.date(*birthday).year
        return "0{:04}{:05}".format(birth_year, Staff._id_num)
    
    def __init__(self, name, sex, birthday, entrydate=None):
        super().__init__(name, sex, birthday,
                         Staff._id_gen(birthday))
        if entrydate:
            self._entrydate = datetime.date(*entrydate)
        else:
            self._entrydate = datetime.date.today()
        self._salary = 1720      # 默认设为最低工资, 可修改
        self._department = None  # 由具体类设定
        self._position   = None  # 由具体类设定

    def set_salary(self, amount):
        if not isinstance(amount, int):
            raise PersonTypeError("Salary should be int.")
        self._salary = amount

    def details(self):
        return ", ".join((super().details(),
                          "年龄: " + str(self.age())))


if __name__ == '__main__':

    def test_Person():
        global plist1

        print("TEST Class Person:")
        
        p1 = Person("谢雨洁", "女", (1995, 7, 30))
        p2 = Person("汪力强", "男", (1990, 2, 17))
        p3 = Person("廖晓溪", "女", (1996, 10, 16))
        p4 = Person("李羽飞", "男", (1994, 5, 24))
        print(p1)
        print(p2)
        print(p3)
        print(p4)
        
        print("\nTest x.details():")
        print(p1.details())

        print("\nTest x.age():")
        print(p1.name(), p1.age(), "years")
        print(p2.name(), p2.age(), "years")

        print("\nTest __str__:")
        print(p1)
        print(p2)

        print("\nAfter sorting:")
        plist1 = [p1, p2, p3, p4]
        plist1.sort()
        for p in plist1:
            print(p)
            print(p.details())

    test_Person()

    def test_UniPerson():
        global plist2

        print("\nTEST Class UniPerson:")
        p1 = UniPerson("谢雨洁", "女", (1995, 7, 30), "1400010111")
        p2 = UniPerson("汪力强", "男", (1990, 2, 17), "1300080324")
        p3 = UniPerson("张子琳", "女", (1974, 10, 16), "0006174132")
        p4 = UniPerson("李国栋", "男", (1962, 5, 24), "0006262018")

        plist2 = [p1, p2, p3, p4]
        for p in plist2:
            print(p)
            print(p.details())
            
        print("\nAfter sorting:")
        plist2.sort()
        for p in plist2:
            print(p)

        print("\nPeople created:", UniPerson.num())

    test_UniPerson()

    def test_Student():
        global student_list

        print("\nTEST Class Undergraduate, MasterStudent, DoctorStudent:")
        p1 = Undergraduate("谢雨洁", "女", (1995, 7, 30), "数学", 3)
        p2 = MasterStudent("汪力强", "男", (1990, 2, 17), "物理")
        p3 = DoctorStudent("李加林", "男", (1989, 4, 10), "化学",
                           "有机化学", "刘小勇")

        print(p1)
        print(p1.details())
        print(p2)
        print(p2.details())
        print(p3)
        print(p3.details())

        student_list = [p1, p2, p3]

    test_Student()

    def test_Staff():
        global staff_list

        print("\nTEST Class Staff:")

        p1 = Staff("张子琳", "女", (1974, 10, 16))
        p2 = Staff("李国栋", "男", (1962, 5, 24))

        print(p1)
        print(p1.details())
        print(p2)
        print(p2.details())

        staff_list = [p1, p2]

    test_Staff()

    print("\nTotal number of the people created:",
          UniPerson.num(), "\n")

    ## 检查属性查询路径
    print("StaffMaster 的属性查询路径:")
    print(StaffMaster.__mro__) # 得到方法检索路径的元组
