## 单链表类 linked_list.py，其中使用链表结点类

from list_node import SLNode

class LinkedListUnderflow(ValueError):
    pass

class LinkedList:
    Node = SLNode
    
    def __init__(self, inits=()):
        self._head = None
        if not inits:
            return
        try:
            it = iter(inits) #  为统一处理序列和迭代器
            p = type(self).Node(next(it))
            self._head = p
            for x in it:
                p.next = type(self).Node(x)
                p = p.next
            return
        except TypeError:
            print("Non-iterable is used in Initiating LinkedList.")
            raise
        except StopIteration: # 空迭代器，直接返回
            return
        
    def __bool__(self):
        return self._head is not None

    def __len__(self):
        p = self._head
        ln = 0
        while p:
            p = p.next
            ln += 1
        return ln

    def __str__(self):
        p = self._head
        s = []
        while p:
            try:
                s.append(str(p.elem))
            except TypeError:
                print("Element in LinkedList can not convert to str.")
                raise
            p = p.next
        return "[[{}]]".format(", ".join(s))

    def __iter__(self): # 用生成器函数定义链表的迭代器
        p = self._head
        while p:
            yield p.elem
            p = p.next

    def __getitem__(self, key):
        # key是整数时返回以key为下标的元素
        # key是切片时构造一个新的LinkedList
        if isinstance(key, int):
            if key < 0:
                raise IndexError("LinkedList does not support"
                                 " neg index", key)
            p = self._head
            i = 0
            while i < key:
                if p is None:
                    raise IndexError("Index in LinkedList is"
                                     " out of range", key)
                p = p.next
                i += 1
            return p.elem
        if isinstance(key, slice):
            res = LinkedList()
            p = self._head
            if not p: return res

            start = key.start or 0
            stop = key.stop
            step = key.step or 1
            print(start, stop, step) # 输出切片描述
            if not (isinstance(start, int) and
                    isinstance(step, int) and
                    (isinstance(stop, int) or stop is None)):
                raise TypeError("Index for slice in LinkedList")
            # 切片中的下标都合法，stop是None表示做到底
            if step <= 0:
                raise ValueError("Step cannot be 0 or less"
                                 " in slice for LinkedList")
            if stop and start >= stop: # 应该得到空表
                return res
            i = 0
            while i < start:
                if p is None: return res
                p = p.next
                i += 1

            newp = res._head = type(self).Node(p.elem) # 新表的首结点

            while p and (stop is None or i < stop):
                num = 0
                while num < step:
                    p = p.next
                    num += 1
                    i += 1
                    if p is None or (stop and i == stop): break
                else:
                    newp.next = type(self).Node(p.elem)
                    newp = newp.next

            return res
        raise TypeError(key, "used as index in LinkedList")
    
    def prepend(self, elem):
        self._head = type(self).Node(elem, self._head)
        
    def prepop(self):
        if self._head is None:
            raise LinkedListUnderflow("prepop empty LinkedList")
        e = self._head.elem
        self._head = self._head.next
        return e

    def append(self, elem):
        if self._head is None:
            self._head = type(self).Node(elem)
            return
        p = self._head
        while p.next:
            p = p.next
        p.next = type(self).Node(elem)

    def pop(self):
        if self._head is None:  # 空表
            raise LinkedListUnderflow("pop empty LinkedList")
        p = self._head
        if p.next is None:  # 一个元素的表
            e = p.elem
            self._head = None
            return e
        while p.next.next:  # 直至p.next是最后结点
            p = p.next
        e = p.next.elem
        p.next = None
        return e

    def reverse(self):
        p = None
        while self._head:
            q = self._head
            self._head = q.next
            q.next = p
            p = q
        self._head = p

    def sort(self):
        p = self._head
        if p is None or p.next is None: # 0或1个元素的表已排序
            return

        rem = p.next
        p.next = None  # 设置已排序段的结束
        while rem:
            q, p = None, self._head
            while p and p.elem <= rem.elem: # 查找rem的插入位置
                q = p
                p = p.next
            # p或空，或指向第一个大于rem元素的结点
            
            if q is None:  # rem应该作为排序段的第一个结点
                self._head = rem
            else:          # rem应该接在q之后
                q.next = rem
            q = rem
            rem = rem.next # rem推进
            q.next = p     # rem插入原p/q所指结点之间，包括p为空情况

    def mutate_elements(self, trans):
        p = self._head
        while p:
            p.elem = trans(p.elem)
            p = p.next

    def __reversed__(self):
        self.reverse()
        try:
            p = self._head
            while p:
                yield p.elem
                p = p.next
        finally:
            self.reverse()
            

#end of class LinkedList


import random

if __name__ == '__main__':
    mlist = LinkedList() # 创建空表
    print(mlist)    
    try:
        mlist.pop()
    except LinkedListUnderflow as ex:
        print(type(ex), ex.args, "\n")

    try:
        mlist = LinkedList(2) # 错误调用，返回空表
    except TypeError as ex:
        print(type(ex), ex.args, "\n")

    mlist = LinkedList("") # 用空序列初始化
    print(mlist)
    mlist = LinkedList("abcd") # 用一般序列初始化
    print(mlist)

    mlist = LinkedList(range(0)) # 用空迭代器初始化
    print(mlist)
    mlist = LinkedList(range(8)) # 用一般迭代器初始化
    print(mlist)

    if mlist: # 链表可以作为条件真值
        print("The list is not empty.")
    else:
        print("The list is empty.")
        
    for i in range(11, 17):
        mlist.prepend(i)
    for i in range(17, 22):
        mlist.append(i)
    print(mlist, "\n")


    print("链表作为可迭代对象用于描述式")
    print([x for x in mlist if x % 2 == 1]) # 链表可以作为可迭代对象
        

    print("mlist[3] is ", mlist[3]) # 能正确处理整数下标
    try:
        print("Try mlist[-5]")
        x = mlist[-5]
    except IndexError as ex:
        print(ex)
    try:
        print("Try mlist[30]")
        x = mlist[30]
    except IndexError as ex:
        print(ex)

    # 处理各种切片形式
    print(mlist[::])
    print(mlist[3::])
    print(mlist[3:30:])
    print(mlist[3::3])
    print(mlist[::3])
    print(mlist[3:16:3])

    # 用于检查 in 和 not in
    if -1 in mlist:
        print("something wrong.")
    elif 4 in mlist:
        print("4 is in mlist.")

    print("\nSome elements are popped.")
    for i in range(4):
        print(mlist.prepop())
        print(mlist.pop())

    print("\nRemained list:", mlist)

    mlist.reverse()
    print("\nReversed list:", mlist)

    mlist.sort()
    print("\nSorted list:", mlist)

    mlist.mutate_elements(lambda x: x * 2)
    print("\nEvery element doubled:", mlist)

    try:
        for x in reversed(mlist):
            if x < 20:
                raise Exception
            else:
                print(x)
    except:
        pass
    
    print(mlist)

