### 一个本科学生登记表类

from personnel import Undergraduate

class Register:
    def __init__(self):
        self._entries = {}

    def __len__(self):
        return len(self._entries)

    def __getitem__(self, sid):
        if not isinstance(sid, str):
            raise TypeError("Student Id")
        if sid not in self._entries:
            raise KeyError("No Student with this Id")
        return self._entries[sid]

    def add(self, student):
        if not isinstance(student, Undergraduate):
            raise TypeError("Not A Student")
        self._entries[student.id()] = student

    def __contains__(self, sid):
        return sid in self._entries

    def __str__(self):
        return "\n".join(self._id_names())

    def __iter__(self):
        for sid in sorted(self._entries):
            yield self._entries[sid]

    def _id_names(self):
        for st in self.__iter__():
            yield st.id() + ', ' + st.name()

    st_display = "学号:{}, 姓名: {:>4}, 性别: {}, 出生年月日: {}\n"\
                 "院系:{:>8}, 入学:  {:6}, 平均成绩: {:4.1f} 共 {} 门课"

    def display(self, sid):
        if sid not in self._entries:
            raise KeyError("No Student with this Id")
        st = self._entries[sid]
        scores = [cs[1] for cs in st.scores()]
        average = sum(scores)/len(scores) if scores else 0
        print(Register.st_display.format(
            st.id(), st.name(), st.sex(), st.birthday(),
            st.department(), st.en_year(), average, len(scores)))

    def record(self, course, scores):
        entries = self._entries
        for sc in scores:
            try:
                entries[sc[0]].set_score(course, sc[1])
            except KeyError as ex:
                print("No Student with this Id:", ex.args[0], "\n")
        
        
## End of Class Register

if __name__ == "__main__":
    import pickle
    
    def basic_test():
        print("Test __str__")
        print(math)

        print("\nTest __iter__")
        for st in math:
            print(st.id(), st.name(), st.age())

        print("\nTest __getitem__")
        for st in math:
            s = math[st.id()]
            print(s.id(), s.name(), s.age(), s.sex())
        print("End of basic test.\n")

    math = Register()

    p1 = Undergraduate("谢雨洁", "女", (1995, 7, 30), "数学", 3)
    p2 = Undergraduate("汪力强", "男", (1994, 2, 17), "数学", 1)
    p3 = Undergraduate("李加林", "男", (1995, 4, 10), "数学", 2)

    p1.set_score("计算概论", 87)
    p2.set_score("计算概论", 90)
    p3.set_score("计算概论", 74)
        
    p1.set_score("数据结构", 85)
    p2.set_score("数据结构", 92)
    p3.set_score("数据结构", 77)

    math.add(p1)
    math.add(p2)
    math.add(p3)

    basic_test()

    math.record("数学分析1",
                [("1201600001", 92), ("1201600001", 88),
                 ("1201600001", 70), ("1000000004", 49)])
    # 上面学生成绩中有一个不存在的学号，检查程序里的异常处理
    math.record("高等代数1",
                [("1201600001", 95), ("1201600001", 90),
                 ("1201600001", 82)])
 
    for st in math:
        math.display(st.id())
        print(st.scores())


    with open('data.pickle', 'wb') as f:
        # Pickle the Register object 'math' using default protocol.
        pickle.dump(math, f)

    with open('data.pickle', 'rb') as f:
        # The protocol is detected automatically. 
        # 把装入重建的对象赋给变量 data.
        data = pickle.load(f)

    print("\nData after pickle and unpickle:")
    for st in data:
        data.display(st.id())
        print(st.scores())


